class FA18_DDI_WaypointManager {
    constructor(ddi) {
        this._initialized = false;
        this.MAIN_FLIGHTPLAN_ID = 0;
        this.MAX_WAYPOINTS = 60;
        this._currentWaypointNumber = 0;
        this.MAX_SEQUENCES = 3;
        this._autoSequentialSteeringActive = false;
        this.doAutoSeqSteerUpdate = false;
        this.previousPlaneCoordinates = new LatLong(0, 0);
        this.activateDirectToRequested = false;
        this.insertionIndex = -1;
        this.ddi = ddi;
    }
    get flightPlanManager() { return this.ddi.flightPlanManager; }
    get isMaster() { return this.ddi.isMaster; }
    get initialized() { return this._initialized; }
    get currentWaypointNumber() { return this._currentWaypointNumber; }
    setCurrentWaypointNumber(n, simvarSet = true) {
        if (this.currentWaypointNumber !== n) {
            this._currentWaypointNumber = n;
            if (simvarSet) {
                SimVar.SetSimVarValue("L:FA18_DDI_CURRENT_WAYPOINT_NUMBER", "number", this.currentWaypointNumber);
                this.sendCurrentWaypointInfosToUfcd();
            }
        }
    }
    get currentWaypoint() { return this.getWaypointFromNumber(this.currentWaypointNumber); }
    get currentWaypointFlightPlanIndex() { return this.getWaypointFlightPlanIndexFromNumber(this.currentWaypointNumber); }
    getWaypointFlightPlanIndexFromNumber(number) {
        if (this.currentSequence && this.currentSequence.selected) {
            return this.currentSequence.getWaypointFlightPlanIndexFromNumber(number);
        }
        if (this.waypointsMapping && !isNaN(this.waypointsMapping[number])) {
            return this.waypointsMapping[number];
        }
        return -1;
    }
    getWaypointFromNumber(number) {
        if (this.waypointsMapping && !isNaN(this.waypointsMapping[number]))
            return this.flightPlanManager.getWaypoint(this.waypointsMapping[number], this.MAIN_FLIGHTPLAN_ID);
        return null;
    }
    getWaypointNumberFromWaypoint(wantedWayPoint) {
        if (this.waypointsMapping) {
            let fpIndex = this.flightPlanManager.getWaypoints(this.MAIN_FLIGHTPLAN_ID).findIndex((wp) => {
                return wp.ident === wantedWayPoint.ident;
            });
            if (fpIndex !== -1) {
                let number = this.waypointsMapping.findIndex((index) => { return index === fpIndex; });
                if (number !== -1)
                    return number;
            }
        }
        return NaN;
    }
    get currentSequenceIndex() { return this._currentSequenceIndex; }
    setCurrentSequenceIndex(id, simvarSet = true) {
        if (this._currentSequenceIndex !== id) {
            this._currentSequenceIndex = id;
            if (simvarSet)
                SimVar.SetSimVarValue("L:FA18_DDI_CURRENT_SEQUENCE_INDEX", "number", this.currentSequenceIndex);
        }
    }
    get currentSequence() {
        if (this.sequences && this._currentSequenceIndex < this.sequences.length)
            return this.sequences[this._currentSequenceIndex];
        else
            return null;
    }
    get autoSequentialSteeringActive() { return this._autoSequentialSteeringActive; }
    // ;
    setAutoSequentialSteeringActive(value, simvarSet = true) {
        if (this.initialized && this.autoSequentialSteeringActive !== value) {
            this._autoSequentialSteeringActive = value;
            if (!value)
                this.doAutoSeqSteerUpdate = false;
            else
                setTimeout(() => { if (this.autoSequentialSteeringActive)
                    this.doAutoSeqSteerUpdate = true; }, 100);
            if (simvarSet)
                SimVar.SetSimVarValue("L:FA18_DDI_AUTO_SEQ_STEERING", "bool", value);
        }
    }
    init() {
        this._initialized = false;
        this.checkMasterInitIntervalId = null;
        this.flightPlanInitStarted = false;
        this.setCurrentWaypointNumber(0, this.isMaster);
        this.setCurrentSequenceIndex(0, this.isMaster);
        if (this.isMaster) {
            SimVar.SetSimVarValue("L:MAP_SHOWN_FLIGHTPLAN_INDEX", "number", this.MAIN_FLIGHTPLAN_ID);
            SimVar.SetSimVarValue("L:VFR_MAP_SHOWN_FLIGHTPLAN_INDEX", "number", this.MAIN_FLIGHTPLAN_ID);
            this.ddi.addEventListener("Reboot", () => {
                if (this.isMaster && this.initialized) {
                    this._initialized = false;
                    this.flightPlanInitStarted = false;
                }
            });
        }
    }
    async flightPlanInit(waypointsMapping = "") {
        this.flightPlanInitStartTime = performance.now();
        this.flightPlanInitStarted = true;
        this.setCurrentWaypointNumber(0, this.isMaster);
        await this.transformWaypointsToCustoms();
        await this.updateWaypointMapping(waypointsMapping);
        await this.initSequences();
        if (this.isMaster)
            await this.flightPlanManager.asyncSetDirectToCancelDistance(-1);
        this._initialized = true;
        if (this.isMaster) {
            this.sendEventToAllSlaves("MasterInitialized", this.formatWaypointsMappingString());
        }
    }
    async transformWaypointsToCustoms() {
        if (this.isMaster && await this.switchToFlightPlan(this.MAIN_FLIGHTPLAN_ID)) {
            await this.flightPlanManager.asyncUpdateFlightPlanByIndex(this.MAIN_FLIGHTPLAN_ID);
            await this.flightPlanManager.asyncUpdateCurrentApproachWaypoints();
            let waypoints = this.flightPlanManager.getWaypoints(this.MAIN_FLIGHTPLAN_ID);
            let approachWaypoints = this.flightPlanManager.getApproachWaypoints();
            waypoints = [...waypoints.slice(0, -1), ...approachWaypoints, ...waypoints.slice(-1)];
            await this.flightPlanManager.asyncClearFlightPlanByIndex(this.MAIN_FLIGHTPLAN_ID, false);
            for (let i = 0; i < waypoints.length; i++) {
                if (waypoints[i]) {
                    await this.flightPlanManager.asyncAddCustomWaypoint("WP" + i, i, waypoints[i].latitudeFP, waypoints[i].longitudeFP, false, false);
                    await this.flightPlanManager.asyncSetWaypointAltitude(waypoints[i].altitudeinFP * Avionics.Utils.FEET2METER, i, false);
                }
            }
        }
        await this.flightPlanManager.asyncUpdateFlightPlanByIndex(this.MAIN_FLIGHTPLAN_ID);
        this.flightPlanManager.updateCurrentApproach();
    }
    async initSequences() {
        if (this.isMaster) {
            await this.flightPlanManager.asyncCreateNewFlightPlansUntilIndex(this.MAX_SEQUENCES);
        }
        this.sequences = [];
        this.setCurrentSequenceIndex(0, this.isMaster);
        for (let i = 0; i < this.MAX_SEQUENCES; i++) {
            let sequence = new FA18_DDI_Waypoint_Sequence(this, i + 1);
            if (sequence) {
                await sequence.init();
                this.sequences.push(sequence);
            }
        }
        if (this.isMaster && this.currentSequence) {
            await this.currentSequence.loadWaypointsToSequence();
        }
    }
    sendEventToAllSlaves(_event, ..._args) {
        this.ddi.triggerEventToMultipleInstruments(this.ddi.slavesInstances, "WaypointManager_" + _event, ..._args);
    }
    sendEventToSlave(_slaveId, _event, ..._args) {
        this.ddi.triggerEventToInstrument(_slaveId, "WaypointManager_" + _event, ..._args);
    }
    sendEventToMaster(_event, ..._args) {
        this.ddi.triggerEventToInstrument(this.ddi.masterInstance, "WaypointManager_" + _event, ..._args);
    }
    sendCurrentWaypointInfosToUfcd() {
        if (this.currentWaypoint)
            this.ddi.triggerEventToInstrument("AS04F_ACC_UFCD", "CurrentSteeringWaypointChanged", this.currentWaypoint.ident, this.currentWaypoint.latitudeFP + '', this.currentWaypoint.longitudeFP + '', (this.currentWaypoint.altitudeinFP * Avionics.Utils.FEET2METER) + '');
    }
    onEvent(_event, _args) {
        switch (_event) {
            case "MasterInitialized":
                if (!this.isMaster && _args.length > 0)
                    this.flightPlanInit(_args[0]);
                break;
            case "CheckMasterInit":
                if (this.isMaster && this.initialized && _args.length > 0 && _args[0])
                    this.sendEventToSlave(_args[0], "MasterInitialized", this.formatWaypointsMappingString());
                break;
            case "WaypointEditSet":
                if (this.isMaster && this.initialized && _args.length > 1)
                    this.onEditWaypoint(_args[0], parseFloat(_args[1]));
                break;
            case "ForceUpdate":
                if (this.initialized)
                    this.updateAllFlightPlans();
                break;
            case "UpdateWaypointMapping":
                if (!this.isMaster && _args.length > 0)
                    this.updateWaypointMapping(_args[0]);
                break;
            case "SequenceInsertWaypoint":
                if (this.isMaster && this.initialized && _args.length > 0)
                    this.onSequenceInsertWaypoint(parseInt(_args[0]));
                break;
            case "SequenceRemoveWaypoint":
                if (this.isMaster && this.initialized && _args.length > 0)
                    this.onSequenceRemoveWaypoint(parseInt(_args[0]));
                break;
            case "UfcdRequestCurrentWaypointInfos":
                if (this.isMaster && this.initialized)
                    this.sendCurrentWaypointInfosToUfcd();
                break;
        }
    }
    formatWaypointsMappingString() {
        let mapping = "";
        if (this.isMaster && this.initialized) {
            for (let i = 0; i < this.waypointsMapping.length; i++) {
                if (!isNaN(this.waypointsMapping[i])) {
                    if (mapping)
                        mapping += ",";
                    mapping += (i + "=" + this.waypointsMapping[i]);
                }
            }
        }
        return mapping;
    }
    async updateWaypointMapping(mapping = "") {
        this.waypointsMapping = new Array(this.MAX_WAYPOINTS).fill(NaN);
        if (this.isMaster) {
            for (let i = 0; i < this.flightPlanManager.getWaypointsCount(this.MAIN_FLIGHTPLAN_ID); i++) {
                this.waypointsMapping[i] = i;
            }
        }
        else if (mapping) {
            let arr = mapping.split(",");
            for (let i = 0; i < arr.length; i++) {
                let x = arr[i].split("=");
                if (x.length == 2) {
                    let n = parseInt(x[0]);
                    let id = parseInt(x[1]);
                    if (!isNaN(n) && !isNaN(id) && n < this.waypointsMapping.length)
                        this.waypointsMapping[n] = id;
                }
            }
        }
    }
    async switchToFlightPlan(flightPlanIndex) {
        if (this.isMaster) {
            if (this.flightPlanManager.getCurrentFlightPlanIndex() != flightPlanIndex)
                return await this.flightPlanManager.asyncSetCurrentFlightPlanIndex(flightPlanIndex);
            return true;
        }
        return false;
    }
    checkInit() {
        if (this.isMaster) {
            if (!this.flightPlanInitStarted && this.ddi.getGameState() >= GameState.briefing) {
                this.flightPlanManager.onCurrentGameFlightLoaded(() => {
                    this.flightPlanInit();
                });
            }
        }
        else {
            if (!this.flightPlanInitStarted) {
                if (!this.checkMasterInitIntervalId)
                    this.checkMasterInitIntervalId = setInterval(() => this.sendEventToMaster("CheckMasterInit", this.ddi.instrumentIdentifier), 2000);
            }
            else if (this.checkMasterInitIntervalId) {
                clearInterval(this.checkMasterInitIntervalId);
                this.checkMasterInitIntervalId = null;
            }
        }
    }
    checkSynchronizedDatas() {
        let synchronizedWaypointNumber = SimVar.GetSimVarValue("L:FA18_DDI_CURRENT_WAYPOINT_NUMBER", "number");
        if (this.currentWaypointNumber !== synchronizedWaypointNumber) {
            this.setCurrentWaypointNumber(synchronizedWaypointNumber, false);
        }
        let synchronizedCurrentSequenceIndex = SimVar.GetSimVarValue("L:FA18_DDI_CURRENT_SEQUENCE_INDEX", "number");
        if (!this.currentSequence || this.currentSequenceIndex !== synchronizedCurrentSequenceIndex) {
            this.setCurrentSequenceIndex(synchronizedCurrentSequenceIndex, false);
        }
        let synchronizedSequenceSelected = SimVar.GetSimVarValue("L:MAP_SHOWN_FLIGHTPLAN_INDEX", "number");
        for (let i = 0; i < this.sequences.length; i++) {
            let shouldBeSelected = (synchronizedSequenceSelected === this.currentSequence.id);
            if (this.sequences[i].selected !== shouldBeSelected)
                this.sequences[i].setSelected(shouldBeSelected, false);
        }
        let synchronizedAutoSequentialSteering = !!SimVar.GetSimVarValue("L:FA18_DDI_AUTO_SEQ_STEERING", "Bool");
        if (this.autoSequentialSteeringActive !== synchronizedAutoSequentialSteering) {
            this.setAutoSequentialSteeringActive(synchronizedAutoSequentialSteering, false);
        }
    }
    async directToUpdate() {
        if (this.isMaster && this.initialized) {
            if (this.ddi.steeringSource === FA18_DDI_Steering_Source.WYPT && this.currentWaypoint != null) {
                let needRefreshDirectTo = !this.flightPlanManager.getIsDirectTo()
                    || (!this.ddi.map.isCourseSteering && (Avionics.Utils.computeGreatCircleDistance(this.previousPlaneCoordinates, this.ddi.planeCoordinates) > 0.1))
                    || (this.currentWaypoint.ident != this.flightPlanManager.getDirectToTarget().ident);
                if (!this.activateDirectToRequested && needRefreshDirectTo) {
                    this.activateDirectToRequested = true;
                    this.previousPlaneCoordinates.lat = this.ddi.planeCoordinates.lat;
                    this.previousPlaneCoordinates.long = this.ddi.planeCoordinates.long;
                    this.flightPlanManager.activateDirectToFromLatLonAlt(this.currentWaypoint.latitudeFP, this.currentWaypoint.longitudeFP, this.currentWaypoint.altitudeinFP, () => { this.activateDirectToRequested = false; }, this.currentWaypoint.ident);
                }
            }
            else if (this.flightPlanManager.getIsDirectTo()) {
                this.flightPlanManager.cancelDirectTo();
            }
        }
    }
    autoSequentialSteeringUpdate() {
        var _a;
        if (this.isMaster && this.initialized && this.doAutoSeqSteerUpdate) {
            if (this.ddi.steeringSource !== FA18_DDI_Steering_Source.WYPT) {
                this.setAutoSequentialSteeringActive(false);
            }
            else {
                if (Math.abs(this.flightPlanManager.getBearingToWaypoint(this.currentWaypoint, true) - Simplane.getHeadingMagnetic()) < 90 && this.flightPlanManager.getDistanceToWaypoint(this.currentWaypoint) < 5) {
                    if (((_a = this.currentSequence) === null || _a === void 0 ? void 0 : _a.waypointsNumber.slice(-1)[0]) === this.currentWaypointNumber) {
                        this.setAutoSequentialSteeringActive(false);
                    }
                    else {
                        this.nextWaypoint();
                    }
                }
            }
        }
    }
    update(deltaTime) {
        if (!this.initialized) {
            this.checkInit();
        }
        else {
            this.directToUpdate();
            this.autoSequentialSteeringUpdate();
            this.checkSynchronizedDatas();
        }
    }
    async updateAllFlightPlans() {
        await this.flightPlanManager.asyncUpdateFlightPlanByIndex(this.MAIN_FLIGHTPLAN_ID);
        for (let i = 0; i < this.sequences.length; i++) {
            await this.sequences[i].update();
        }
    }
    setCourseSteering(_enabled) {
        if (this.isMaster)
            this.ddi.map.onEvent("CRS_Set_Steering", [_enabled ? '1' : '0']);
        else
            this.ddi.triggerEventToInstrument(this.ddi.masterInstance, "Map_CRS_Set_Steering", _enabled ? '1' : '0');
    }
    nextWaypoint() {
        if (this.initialized) {
            if (this.currentSequence && this.currentSequence.selected) {
                let nextWaypointNumber = this.currentSequence.nextWaypointNumber;
                if (nextWaypointNumber != -1)
                    this.setCurrentWaypointNumber(nextWaypointNumber);
            }
            else {
                if (this.currentWaypointNumber < (this.MAX_WAYPOINTS - 1))
                    this.setCurrentWaypointNumber(this.currentWaypointNumber + 1);
                else
                    this.setCurrentWaypointNumber(0);
            }
            this.setCourseSteering(false);
        }
    }
    previousWaypoint() {
        if (this.initialized) {
            if (this.currentSequence && this.currentSequence.selected) {
                let previousWaypointNumber = this.currentSequence.previousWaypointNumber;
                if (previousWaypointNumber != -1)
                    this.setCurrentWaypointNumber(previousWaypointNumber);
            }
            else {
                if (this.currentWaypointNumber > 0)
                    this.setCurrentWaypointNumber(this.currentWaypointNumber - 1);
                else
                    this.setCurrentWaypointNumber(this.MAX_WAYPOINTS - 1);
            }
            this.setCourseSteering(false);
        }
    }
    toggleSequences() {
        if (this.initialized && this.currentSequence) {
            if (this.currentSequence.selected) {
                this.currentSequence.setSelected(false);
                this.insertionIndex = -1;
                if ((this.currentSequenceIndex + 1) < this.sequences.length)
                    this.setCurrentSequenceIndex(this.currentSequenceIndex + 1);
                else
                    this.setCurrentSequenceIndex(0);
                if (this.autoSequentialSteeringActive)
                    this.engageAutoSeqSteering();
            }
            else {
                this.currentSequence.setSelected(true);
            }
        }
    }
    toggleAutoSequencialSteering() {
        if (this.initialized && this.currentSequence) {
            if (this.autoSequentialSteeringActive) {
                this.setAutoSequentialSteeringActive(false);
            }
            else {
                this.engageAutoSeqSteering();
            }
        }
    }
    engageAutoSeqSteering() {
        if (this.currentSequence.waypointsNumber.length > 1) {
            this.ddi.setSteeringSource(FA18_DDI_Steering_Source.WYPT);
            if (!this.currentSequence.selected)
                this.currentSequence.setSelected(true);
            if (this.currentWaypointNumber !== this.currentSequence.waypointsNumber[0])
                this.setCurrentWaypointNumber(this.currentSequence.waypointsNumber[0]);
            this.setAutoSequentialSteeringActive(true);
        }
        else {
            this.setAutoSequentialSteeringActive(false);
        }
    }
    onEditWaypoint(mode, value) {
        if (this.isMaster && this.initialized && mode && !isNaN(value)) {
            switch (mode) {
                case "Latitude":
                    this.editWaypoint(value, null, null);
                    break;
                case "Longitude":
                    this.editWaypoint(null, value, null);
                    break;
                case "Altitude":
                    this.editWaypoint(null, null, value);
                    break;
            }
        }
    }
    async editWaypoint(latitude, longitude, altitude, waypointNumber = NaN) {
        if (this.isMaster && this.initialized) {
            let currentFlightPlanId = this.flightPlanManager.getCurrentFlightPlanIndex();
            await this.switchToFlightPlan(this.MAIN_FLIGHTPLAN_ID);
            if (isNaN(waypointNumber)) {
                waypointNumber = this.currentWaypointNumber;
            }
            let wp = this.getWaypointFromNumber(waypointNumber);
            if (wp) {
                await this.flightPlanManager.asyncSetWaypointLatLonAlt(this.getWaypointFlightPlanIndexFromNumber(waypointNumber), (latitude != null) ? latitude : wp.latitudeFP, (longitude != null) ? longitude : wp.longitudeFP, ((altitude != null) ? altitude : wp.altitudeinFP) * Avionics.Utils.FEET2METER, false);
                for (let i = 0; i < this.sequences.length; i++) {
                    await this.sequences[i].editWaypoint(waypointNumber, wp.latitudeFP, wp.longitudeFP, wp.altitudeinFP, false);
                }
            }
            else {
                let insertionIndex = 0;
                let newWaypointsMapping = [...this.waypointsMapping];
                for (let i = 0; i < this.waypointsMapping.length; i++) {
                    if (!isNaN(this.waypointsMapping[i])) {
                        if (i < waypointNumber)
                            insertionIndex++;
                        else
                            newWaypointsMapping[i] = this.waypointsMapping[i] + 1;
                    }
                }
                newWaypointsMapping[waypointNumber] = insertionIndex;
                await this.flightPlanManager.asyncAddCustomWaypoint("WP" + waypointNumber, insertionIndex, (latitude != null) ? latitude : 0, (longitude != null) ? longitude : 0, false, false);
                if (altitude != null) {
                    await this.flightPlanManager.asyncSetWaypointAltitude(altitude * Avionics.Utils.FEET2METER, insertionIndex, false);
                }
                await this.flightPlanManager.asyncUpdateFlightPlanByIndex(this.MAIN_FLIGHTPLAN_ID);
                this.waypointsMapping = newWaypointsMapping;
                this.sendEventToAllSlaves("UpdateWaypointMapping", this.formatWaypointsMappingString());
            }
            await this.switchToFlightPlan(currentFlightPlanId);
            this.sendEventToAllSlaves("ForceUpdate");
        }
    }
    async onSequenceInsertWaypoint(waypointNumber) {
        if (this.isMaster && this.initialized && !isNaN(waypointNumber) && this.currentSequence) {
            let insertAfterWaypointIndex = this.currentSequence.getWaypointFlightPlanIndexFromNumber(waypointNumber);
            if (insertAfterWaypointIndex !== -1) {
                this.insertionIndex = insertAfterWaypointIndex + 1;
            }
            else {
                if (this.getWaypointFromNumber(waypointNumber) == null) {
                    await this.editWaypoint(null, null, null, waypointNumber);
                }
                await this.currentSequence.insertWaypoint(waypointNumber, this.insertionIndex);
                this.insertionIndex = -1;
                this.sendEventToAllSlaves("ForceUpdate");
            }
        }
    }
    async onSequenceRemoveWaypoint(waypointNumber) {
        if (this.isMaster && this.initialized && !isNaN(waypointNumber) && this.currentSequence) {
            await this.currentSequence.removeWaypoint(waypointNumber);
            this.sendEventToAllSlaves("ForceUpdate");
        }
    }
}
class FA18_DDI_Waypoint_Sequence {
    constructor(_manager, _id) {
        this.MAX_WAYPOINTS_PER_SEQUENCE = 15;
        this._selected = false;
        this.manager = _manager;
        this._id = _id;
    }
    get flightPlanManager() { return this.manager.flightPlanManager; }
    get id() { return this._id; }
    get waypointsNumber() { return this._waypointsNumber; }
    get selected() { return this._selected; }
    setSelected(value, simvarSet = true) {
        if (this.selected !== value) {
            this._selected = value;
            if (simvarSet) {
                if (value) {
                    this.flightPlanManager.setCurrentFlightPlanIndex(this.id, false, () => {
                        SimVar.SetSimVarValue("L:MAP_SHOWN_FLIGHTPLAN_INDEX", "number", this.id);
                        SimVar.SetSimVarValue("L:VFR_MAP_SHOWN_FLIGHTPLAN_INDEX", "number", this.id);
                    });
                }
                else {
                    this.flightPlanManager.setCurrentFlightPlanIndex(this.manager.MAIN_FLIGHTPLAN_ID, false, () => {
                        SimVar.SetSimVarValue("L:MAP_SHOWN_FLIGHTPLAN_INDEX", "number", this.manager.MAIN_FLIGHTPLAN_ID);
                        SimVar.SetSimVarValue("L:VFR_MAP_SHOWN_FLIGHTPLAN_INDEX", "number", this.manager.MAIN_FLIGHTPLAN_ID);
                    });
                }
            }
        }
    }
    getWaypointFlightPlanIndexFromNumber(number) {
        return this.waypointsNumber.findIndex(n => { return n === number; });
    }
    async init() {
        await this.reset();
    }
    async reset() {
        if (this.manager.isMaster)
            await this.flightPlanManager.asyncClearFlightPlanByIndex(this.id);
        this.setSelected(false, false);
        await this.update();
    }
    async loadWaypointsToSequence() {
        if (this.manager.isMaster) {
            let waypoints = this.flightPlanManager.getWaypoints(this.manager.MAIN_FLIGHTPLAN_ID);
            for (let i = 0; i < waypoints.length && i < this.MAX_WAYPOINTS_PER_SEQUENCE; i++) {
                await this.insertWaypoint(i, i, false, false);
            }
            await this.manager.switchToFlightPlan(this.manager.MAIN_FLIGHTPLAN_ID);
            await this.update();
        }
    }
    async update() {
        await this.flightPlanManager.asyncUpdateFlightPlanByIndex(this.id);
        this._waypointsNumber = [];
        for (let i = 0; i < this.flightPlanManager.getWaypointsCount(this.id); i++) {
            this._waypointsNumber.push(this.manager.getWaypointNumberFromWaypoint(this.flightPlanManager.getWaypoint(i, this.id)));
        }
    }
    get nextWaypointNumber() {
        let index = this.waypointsNumber.findIndex(n => { return n === this.manager.currentWaypointNumber; });
        if ((index + 1) < this.waypointsNumber.length)
            return this.waypointsNumber[index + 1];
        if (this.waypointsNumber.length > 0)
            return this.waypointsNumber[0];
        return -1;
    }
    get previousWaypointNumber() {
        let index = this.waypointsNumber.findIndex(n => { return n === this.manager.currentWaypointNumber; });
        if (index > 0)
            return this.waypointsNumber[index - 1];
        if (this.waypointsNumber.length > 0)
            return this.waypointsNumber[this.waypointsNumber.length - 1];
        return -1;
    }
    async insertWaypoint(waypointNumber, index = -1, switchBackToCurrentFpId = true, doUpdate = true) {
        if (this.manager.isMaster) {
            let currentFpId = this.flightPlanManager.getCurrentFlightPlanIndex();
            if (await this.manager.switchToFlightPlan(this.id)) {
                let wp = this.manager.getWaypointFromNumber(waypointNumber);
                if (wp) {
                    if (this.flightPlanManager.getWaypointsCount(this.id) == this.MAX_WAYPOINTS_PER_SEQUENCE) {
                        await this.removeWaypoint(this.waypointsNumber[0], false, false);
                        this.waypointsNumber.splice(0, 1);
                        index--;
                    }
                    if (index < 0)
                        index = this.waypointsNumber.length;
                    await this.flightPlanManager.asyncAddCustomWaypoint(wp.ident, index, wp.latitudeFP, wp.longitudeFP, false, false);
                    await this.flightPlanManager.asyncSetWaypointAltitude(wp.altitudeinFP * Avionics.Utils.FEET2METER, index, false);
                    if (doUpdate)
                        await this.update();
                }
                if (switchBackToCurrentFpId)
                    await this.manager.switchToFlightPlan(currentFpId);
            }
        }
    }
    async editWaypoint(waypointNumber, latitude, longitude, altitude, switchBackToCurrentFpId = true) {
        if (this.manager.isMaster && this.manager.initialized && this.waypointsNumber.includes(waypointNumber)) {
            let wp = this.manager.getWaypointFromNumber(waypointNumber);
            if (wp) {
                let currentFpId = this.flightPlanManager.getCurrentFlightPlanIndex();
                await this.manager.switchToFlightPlan(this.id);
                await this.flightPlanManager.asyncSetWaypointLatLonAlt(this.getWaypointFlightPlanIndexFromNumber(waypointNumber), wp.latitudeFP, wp.longitudeFP, wp.altitudeinFP * Avionics.Utils.FEET2METER, false);
                if (switchBackToCurrentFpId)
                    await this.manager.switchToFlightPlan(currentFpId);
            }
        }
    }
    async removeWaypoint(waypointNumber, switchBackToCurrentFpId = true, doUpdate = true) {
        if (this.manager.isMaster && this.manager.initialized && this.waypointsNumber.includes(waypointNumber)) {
            let currentFpId = this.flightPlanManager.getCurrentFlightPlanIndex();
            if (await this.manager.switchToFlightPlan(this.id)) {
                await this.flightPlanManager.asyncRemoveWaypoint(this.getWaypointFlightPlanIndexFromNumber(waypointNumber), false, false);
                if (doUpdate)
                    await this.update();
                if (switchBackToCurrentFpId)
                    await this.manager.switchToFlightPlan(currentFpId);
            }
        }
    }
}
//# sourceMappingURL=FA18_DDI_WaypointManager.js.map