class FA18ACC_DDI_FPAS extends FA18_DDI_Page {
    constructor() {
        super(...arguments);
        this.fuelReference = 2000;
    }
    get templateID() { return "FA18ACC_DDI_FPAS"; }
    init(ddi) {
        super.init(ddi);
        this.currentRangeTo2000LbElement = this.ddi.getChildById("fpas-current-range-2000lb");
        this.currentRangeBestMachElement = this.ddi.getChildById("fpas-current-range-best-mach");
        this.currentRangeBestMachTo2000LbElement = this.ddi.getChildById("fpas-current-range-best-mach-2000lb");
        this.currentEnduranceTo2000LbElement = this.ddi.getChildById("fpas-current-endurance-2000lb");
        this.currentEnduranceBestMachElement = this.ddi.getChildById("fpas-current-endurance-best-mach");
        this.currentEnduranceBestMachTo2000LbElement = this.ddi.getChildById("fpas-current-endurance-best-mach-2000lb");
        this.navToIdentElement = this.ddi.getChildById("fpas-nav-to-ident");
        this.navToTimeElement = this.ddi.getChildById("fpas-nav-to-time");
        this.navToRemainingFuelElement = this.ddi.getChildById("fpas-nav-to-remaining-fuel");
        this.fuelFlowElement = this.ddi.getChildById("fpas-fuel-flow");
        this.optimumRangeAltitudeElement = this.ddi.getChildById("fpas-optimum-range-altitude");
        this.optimumRangeMachElement = this.ddi.getChildById("fpas-optimum-range-mach");
        this.optimumRangeTo2000LbElement = this.ddi.getChildById("fpas-optimum-range-2000lb");
        this.optimumEnduranceAltitudeElement = this.ddi.getChildById("fpas-optimum-endurance-altitude");
        this.optimumEnduranceMachElement = this.ddi.getChildById("fpas-optimum-endurance-mach");
        this.optimumEnduranceTo2000LbElement = this.ddi.getChildById("fpas-optimum-endurance-2000lb");
        this.toLbsLegendElements = this.getElementsByClassName("fpas-to-lbs");
        this.initDataTables();
    }
    initDataTables() {
        if (this.ddi.xmlConfig) {
            let tableContainer = this.ddi.xmlConfig.getElementsByTagName("AS04F_DDI_FPAS");
            if (tableContainer.length > 0) {
                let nmPerPoundElement = tableContainer[0].getElementsByTagName("SpecificRangeNmPerPound");
                if (nmPerPoundElement.length > 0) {
                    this.specificRangeNmPerPoundTable = new MultiDimensionsTableLogicXMLElement(this.ddi, nmPerPoundElement[0]);
                }
                let bestMachForRangeElement = tableContainer[0].getElementsByTagName("BestMachForRange");
                if (bestMachForRangeElement.length > 0) {
                    this.bestMachForRangeTable = new MultiDimensionsTableLogicXMLElement(this.ddi, bestMachForRangeElement[0]);
                }
                let bestMachForEnduranceElement = tableContainer[0].getElementsByTagName("BestMachForEndurance");
                if (bestMachForEnduranceElement.length > 0) {
                    this.bestMachForEnduranceTable = new MultiDimensionsTableLogicXMLElement(this.ddi, bestMachForEnduranceElement[0]);
                }
                let maximumEnduranceAltitudeElement = tableContainer[0].getElementsByTagName("MaximumEnduranceAltitude");
                if (maximumEnduranceAltitudeElement.length > 0) {
                    this.maximumEnduranceAltitudeTable = new MultiDimensionsTableLogicXMLElement(this.ddi, maximumEnduranceAltitudeElement[0]);
                }
            }
        }
    }
    show() {
        super.show();
        for (let i of [16, 17, 20]) {
            this.ddi.setPageButton(i, FA18_DDI_Page_Type.NotImplemented);
        }
        this.ddi.setPageButton(18, FA18_DDI_Page_Type.MenuTactical);
    }
    updateCurrentRangeAndEnduranceArea() {
        if (this.isSimOnGround) {
            diffAndSetText(this.currentRangeTo2000LbElement, "XXXX");
            diffAndSetHTML(this.currentRangeBestMachElement, "<br/>");
            diffAndSetHTML(this.currentRangeBestMachTo2000LbElement, "<br/>");
            diffAndSetText(this.currentEnduranceTo2000LbElement, "FPAS");
            diffAndSetHTML(this.currentEnduranceBestMachElement, "<br/>");
            diffAndSetHTML(this.currentEnduranceBestMachTo2000LbElement, "<br/>");
        }
        else {
            if (this.currentMach > 0.9) {
                diffAndSetText(this.currentRangeTo2000LbElement, "MACH");
                diffAndSetText(this.currentEnduranceTo2000LbElement, "LIM");
            }
            else {
                diffAndSetText(this.currentRangeTo2000LbElement, this.formatRange(this.computeRange(this.currentNmPerPound)));
                diffAndSetText(this.currentEnduranceTo2000LbElement, this.formatEndurance(this.computeEndurance(this.computePoundsPerHour(this.currentMach, this.currentOAT, this.currentNmPerPound))));
            }
            let bestMachForRange = this.getBestMachForRange(this.currentAltitude, this.currentWeight);
            diffAndSetText(this.currentRangeBestMachElement, this.formatMach(bestMachForRange));
            diffAndSetText(this.currentRangeBestMachTo2000LbElement, this.formatRange(this.computeRange(this.getSpecificRangeNmPerPound(this.currentAltitude, this.currentWeight, bestMachForRange))));
            let bestMachForEndurance = this.getBestMachForEndurance(this.currentAltitude, this.currentWeight);
            diffAndSetText(this.currentEnduranceBestMachElement, this.formatMach(bestMachForEndurance));
            let nmPerPoundForEndurance = this.getSpecificRangeNmPerPound(this.currentAltitude, this.currentWeight, bestMachForEndurance);
            diffAndSetText(this.currentEnduranceBestMachTo2000LbElement, this.formatEndurance(this.computeEndurance(this.computePoundsPerHour(this.currentMach, this.currentOAT, nmPerPoundForEndurance))));
        }
    }
    updateNavToAera(deltaTime) {
        let ident;
        let distance;
        if (!this.isSimOnGround) {
            switch (SimVar.GetSimVarValue("L:FA18_DDI_STEERING_SOURCE", "number")) {
                case FA18_DDI_Steering_Source.WYPT:
                    if (this.ddi.waypointManager.initialized && this.ddi.waypointManager.currentWaypoint) {
                        ident = "WYPT " + this.ddi.waypointManager.currentWaypointNumber;
                        distance = this.ddi.flightPlanManager.getDistanceToWaypoint(this.ddi.waypointManager.currentWaypoint);
                    }
                    break;
                case FA18_DDI_Steering_Source.TACAN:
                    if (this.ddi.tacanAvailable) {
                        ident = SimVar.GetSimVarValue("TACAN STATION IDENT:1", "string");
                        distance = SimVar.GetSimVarValue("TACAN STATION DISTANCE:1", "nautical mile");
                    }
                    break;
				case FA18_DDI_Steering_Source.ILST:
                    if (this.ddi.ilstacanAvailable) {
                        ident = SimVar.GetSimVarValue("NAV IDENT:" + this.ddi.ilstacanAvailable, "string");
                        distance = SimVar.GetSimVarValue("NAV DME:" + this.ddi.ilstacanAvailable, "nautical mile");
                    }
                    break;
				case FA18_DDI_Steering_Source.VOR:
                    if (this.ddi.navAvailable) {
                        ident = SimVar.GetSimVarValue("NAV IDENT:" + this.ddi.navindex, "string");
                        distance = SimVar.GetSimVarValue("NAV DME:" + this.ddi.navindex, "nautical mile");
                    }
                    break;
				case FA18_DDI_Steering_Source.ILSN:
                    if (this.ddi.ilsnavAvailable) {
                        ident = SimVar.GetSimVarValue("NAV IDENT:" + this.ddi.ilsnavAvailable, "string");
                        distance = SimVar.GetSimVarValue("NAV DME:" + this.ddi.ilsnavAvailable, "nautical mile");
                    }
                    break;
            }
        }
        if (ident) {
            diffAndSetText(this.navToIdentElement, ident);
            let timeAtArrivalInSeconds;
            let fuelRemainingAtArrival;
            if (distance != null) {
                let ete = Avionics.Utils.computeETEinSeconds(distance);
                if (ete != null) {
                    timeAtArrivalInSeconds = (Simplane.getCurrentUTC() + ete) % (3600 * 24);
                }
                if (this.currentNmPerPound != null && !isNaN(this.currentNmPerPound)) {
                    let fuelUsed = distance * (1 / this.currentNmPerPound);
                    fuelRemainingAtArrival = Math.max(this.fuelRemaining - fuelUsed, 0);
                }
            }
            if (this.navToIdent != ident) {
                this.navToArrivalTime = timeAtArrivalInSeconds;
                this.navToArrivalRemainingFuel = fuelRemainingAtArrival;
                this.navToIdent = ident;
            }
            else {
                this.navToArrivalTime = Utils.SmoothPow(this.navToArrivalTime, timeAtArrivalInSeconds, 1.1, deltaTime / 1000);
                this.navToArrivalRemainingFuel = Utils.SmoothPow(this.navToArrivalRemainingFuel, fuelRemainingAtArrival, 1.1, deltaTime / 1000);
            }
            if (this.navToArrivalTime != null) {
                diffAndSetText(this.navToTimeElement, Utils.SecondsToDisplayTime(this.navToArrivalTime, true, true));
            }
            else {
                diffAndSetText(this.navToTimeElement, "XXX");
            }
            if (this.navToArrivalRemainingFuel != null) {
                diffAndSetText(this.navToRemainingFuelElement, fastToFixed(this.navToArrivalRemainingFuel, 0));
            }
            else {
                diffAndSetText(this.navToRemainingFuelElement, "XXX");
            }
        }
        else {
            diffAndSetHTML(this.navToIdentElement, "<br/>");
            diffAndSetHTML(this.navToRemainingFuelElement, "<br/>");
            diffAndSetHTML(this.navToTimeElement, "<br/>");
        }
    }
    updateFuelFlowArea() {
        if (this.isSimOnGround)
            diffAndSetHTML(this.fuelFlowElement, "<br/>");
        else
            diffAndSetText(this.fuelFlowElement, (this.currentFuelFlowPph / Simplane.getGroundSpeed()).toFixed(0));
    }
    updateOptimumRange() {
        let targetAltitude = this.computeOptimumRangeAltitude(this.currentWeight);
        let targetMach = this.computeOptimumRangeMach(this.currentWeight);
        diffAndSetText(this.optimumRangeAltitudeElement, this.formatAltitude(targetAltitude));
        diffAndSetText(this.optimumRangeMachElement, this.formatMach(targetMach));
        diffAndSetText(this.optimumRangeTo2000LbElement, this.formatRange(this.computeRange(this.getSpecificRangeNmPerPound(targetAltitude, this.currentWeight, targetMach))));
    }
    updateOptimumEndurance() {
        let targetAltitude = this.getMaximumEnduranceAltitude(this.currentWeight);
        let targetMach = this.getBestMachForEndurance(targetAltitude, this.currentWeight);
        diffAndSetText(this.optimumEnduranceAltitudeElement, this.formatAltitude(targetAltitude));
        diffAndSetText(this.optimumEnduranceMachElement, this.formatMach(targetMach));
        let nmPerPoundForEndurance = this.getSpecificRangeNmPerPound(targetAltitude, this.currentWeight, targetMach);
        let pphForEndurance = this.computePoundsPerHour(targetMach, this.computeOatAtAltitude(targetAltitude, this.mslOAT), nmPerPoundForEndurance);
        diffAndSetText(this.optimumEnduranceTo2000LbElement, this.formatEndurance(this.computeEndurance(pphForEndurance)));
    }
    update(deltaTime) {
        super.update(deltaTime);
        this.isSimOnGround = SimVar.GetSimVarValue("SIM ON GROUND", "Bool");
        this.fuelRemaining = SimVar.GetSimVarValue("FUEL TOTAL QUANTITY", "gallons") * SimVar.GetSimVarValue("FUEL WEIGHT PER GALLON", "lbs");
        this.currentAltitude = Simplane.getAltitude();
        this.currentWeight = Simplane.getWeight() * 2.204623;
        this.currentOAT = Simplane.getAmbientTemperature();
        this.mslOAT = SimVar.GetSimVarValue("SEA LEVEL AMBIENT TEMPERATURE", "celsius");
        this.currentFuelFlowPph = SimVar.GetSimVarValue("TURB ENG FUEL FLOW PPH:1", "pounds per hour") + SimVar.GetSimVarValue("TURB ENG FUEL FLOW PPH:2", "pounds per hour");
        this.currentMach = Simplane.getMachSpeed();
        this.currentNmPerPound = this.getSpecificRangeNmPerPound(this.currentAltitude, this.currentWeight, this.currentMach);
        this.fuelReference = (this.fuelRemaining < 2500) ? 0 : 2000;
        for (let i = 0; i < this.toLbsLegendElements.length; i++) {
            diffAndSetText(this.toLbsLegendElements[i], "TO " + this.fuelReference + " LB");
        }
        this.updateCurrentRangeAndEnduranceArea();
        this.updateNavToAera(deltaTime);
        this.updateFuelFlowArea();
        this.updateOptimumRange();
        this.updateOptimumEndurance();
    }
    formatAltitude(altitude) {
        if (isNaN(altitude))
            return "XXX";
        return fastToFixed(altitude, 0);
    }
    formatMach(mach) {
        if (isNaN(mach))
            return "XXX";
        return fastToFixed(mach, 2).replace(/0*\./g, ".");
    }
    formatEndurance(time) {
        if (isNaN(time))
            return "XXX";
        let h = Math.floor(time);
        let m = (time - h) * 60;
        if (h > 0)
            return fastToFixed(h, 0) + ":" + Utils.leadingZeros(m, 2, 0);
        return ":" + Utils.leadingZeros(m, 2, 0);
    }
    formatRange(rangeValue) {
        if (isNaN(rangeValue))
            return "XXX";
        return fastToFixed(rangeValue, 0);
    }
    computeRange(nmPerPoundValue) {
        if (isNaN(nmPerPoundValue))
            return NaN;
        return nmPerPoundValue * (this.fuelRemaining - this.fuelReference);
    }
    computeEndurance(pphValue) {
        if (isNaN(pphValue))
            return NaN;
        return (this.fuelRemaining - this.fuelReference) / pphValue;
    }
    getSpecificRangeNmPerPound(altitude, weight, mach) {
        if (this.specificRangeNmPerPoundTable && !isNaN(altitude) && !isNaN(weight) && !isNaN(mach)) {
            this.specificRangeNmPerPoundTable.overrideInputsValue(altitude, weight, mach);
            return this.specificRangeNmPerPoundTable.getValue();
        }
        return NaN;
    }
    computeOatAtAltitude(targetAltitude, mslOat) {
        if (!isNaN(targetAltitude) && !isNaN(mslOat)) {
            targetAltitude *= Avionics.Utils.FEET2METER / 1000;
            let temp = 0;
            if (targetAltitude < 11)
                temp = 15 - (6.5 * targetAltitude);
            else if (targetAltitude < 20)
                temp = -56.5;
            else if (targetAltitude < 32)
                temp = -56.5 + targetAltitude;
            else if (targetAltitude < 47)
                temp = -44.5 + 2.8 * (targetAltitude - 32);
            else if (targetAltitude < 51)
                temp = -2.5;
            else if (targetAltitude < 71)
                temp = -2.5 - 2.8 * (targetAltitude - 51);
            else if (targetAltitude < 84.852)
                temp = -58.5 - 2.0 * (targetAltitude - 71);
            else {
                console.warn("ComputeTemperature was given Abnormal GeoHeight (>84Km).");
                temp = 15.0;
            }
            if (temp < -273.15 || temp > 226.85) {
                console.warn("Extreme Temperature Value computed.");
                temp = Utils.Clamp(temp, -273.15, 226.85);
            }
            return temp + mslOat - 15;
        }
        return NaN;
    }
    computePoundsPerHour(mach, oat, nmPerPound) {
        if (!isNaN(mach) && !isNaN(oat) && !isNaN(nmPerPound)) {
            let tas = this.computeTasFromMachAndOat(mach, oat);
            if (!isNaN(tas)) {
                return this.computePphFromTasAndNmPerPound(tas, nmPerPound);
            }
        }
        return NaN;
    }
    computeTasFromMachAndOat(mach, oat) {
        if (!isNaN(mach) && !isNaN(oat)) {
            if (oat < -80)
                return (535.012 * mach - 0.130577);
            if (oat < -60)
                return (((oat + 80) / (-60 + 80)) * ((564.914 * mach - 0.11066) - (535.012 * mach - 0.130577)) + (535.012 * mach - 0.130577));
            if (oat < -40)
                return (((oat + 60) / (-40 + 60)) * ((588.188 * mach + 1.20198) - (564.914 * mach - 0.11066)) + (564.914 * mach - 0.11066));
            if (oat < -20)
                return (((oat + 40) / (-20 + 40)) * ((624.653 * mach - 3.12417) - (588.188 * mach + 1.20198)) + (588.188 * mach + 1.20198));
            if (oat < 0)
                return (((oat + 20) / (0 + 20)) * ((650.092 * mach - 10.1517) - (624.653 * mach - 3.12417)) + (624.653 * mach - 3.12417));
            if (oat < 10)
                return (((oat - 0) / (10 - 0)) * ((654.875 * mach - 4.32602) - (650.092 * mach - 10.1517)) + (650.092 * mach - 10.1517));
            if (oat < 20)
                return (((oat - 10) / (20 - 10)) * ((666.442 * mach - 3.05555) - (654.875 * mach - 4.32602)) + (654.875 * mach - 4.32602));
            if (oat < 40)
                return (((oat - 20) / (40 - 20)) * ((684.619 * mach - 0.653908) - (666.442 * mach - 3.05555)) + (666.442 * mach - 3.05555));
            else
                return (684.619 * mach - 0.653908);
        }
        return NaN;
    }
    computePphFromTasAndNmPerPound(tas, nmPerPound) {
        if (!isNaN(tas) && !isNaN(nmPerPound)) {
            if (nmPerPound < 0.0200)
                return (50.4393 * tas - 116.605);
            if (nmPerPound < 0.0225)
                return (((nmPerPound - 0.0200) / (0.0225 - 0.0200)) * ((44.727 * tas - 65.277) - (50.4393 * tas - 116.605)) + (50.4393 * tas - 116.605));
            if (nmPerPound < 0.0250)
                return (((nmPerPound - 0.0225) / (0.0250 - 0.0225)) * ((40.0919 * tas - 19.5055) - (44.727 * tas - 65.277)) + (44.727 * tas - 65.277));
            if (nmPerPound < 0.0275)
                return (((nmPerPound - 0.0250) / (0.0275 - 0.0250)) * ((36.862 * tas - 181.952) - (40.0919 * tas - 19.5055)) + (40.0919 * tas - 19.5055));
            if (nmPerPound < 0.0300)
                return (((nmPerPound - 0.0275) / (0.0300 - 0.0275)) * ((33.6017 * tas - 127.609) - (36.862 * tas - 181.952)) + (36.862 * tas - 181.952));
            if (nmPerPound < 0.0350)
                return (((nmPerPound - 0.0300) / (0.0350 - 0.0300)) * ((29.2055 * tas - 157.631) - (33.6017 * tas - 127.609)) + (33.6017 * tas - 127.609));
            if (nmPerPound < 0.0400)
                return (((nmPerPound - 0.0350) / (0.0400 - 0.0350)) * ((24.8589 * tas + 74.7359) - (29.2055 * tas - 157.631)) + (29.2055 * tas - 157.631));
            if (nmPerPound < 0.0450)
                return (((nmPerPound - 0.0400) / (0.0450 - 0.0400)) * ((22.0215 * tas + 138.442) - (24.8589 * tas + 74.7359)) + (24.8589 * tas + 74.7359));
            if (nmPerPound < 0.0500)
                return (((nmPerPound - 0.0450) / (0.0500 - 0.0450)) * ((21.0094 * tas - 500.902) - (22.0215 * tas + 138.442)) + (22.0215 * tas + 138.442));
            if (nmPerPound < 0.0550)
                return (((nmPerPound - 0.0500) / (0.0550 - 0.0500)) * ((18.5049 * tas - 65.8881) - (21.0094 * tas - 500.902)) + (21.0094 * tas - 500.902));
            if (nmPerPound < 0.0600)
                return (((nmPerPound - 0.0550) / (0.0600 - 0.0550)) * ((16.6478 * tas + 59.7262) - (18.5049 * tas - 65.8881)) + (18.5049 * tas - 65.8881));
            if (nmPerPound < 0.0700)
                return (((nmPerPound - 0.0600) / (0.0700 - 0.0600)) * ((14.5217 * tas + 9.33967) - (16.6478 * tas + 59.7262)) + (16.6478 * tas + 59.7262));
            if (nmPerPound < 0.0800)
                return (((nmPerPound - 0.0700) / (0.0800 - 0.0700)) * ((12.4997 * tas + 6.67179) - (14.5217 * tas + 9.33967)) + (14.5217 * tas + 9.33967));
            if (nmPerPound < 0.1000)
                return (((nmPerPound - 0.0800) / (0.1000 - 0.0800)) * ((9.97857 * tas + 11.4273) - (12.4997 * tas + 6.67179)) + (12.4997 * tas + 6.67179));
            if (nmPerPound < 0.1200)
                return (((nmPerPound - 0.1000) / (0.1200 - 0.1000)) * ((8.20235 * tas + 69.925) - (9.97857 * tas + 11.4273)) + (9.97857 * tas + 11.4273));
            if (nmPerPound < 0.1600)
                return (((nmPerPound - 0.1200) / (0.1600 - 0.1200)) * ((6.31891 * tas + 26.2884) - (8.20235 * tas + 69.925)) + (8.20235 * tas + 69.925));
            if (nmPerPound < 0.2000)
                return (((nmPerPound - 0.1600) / (0.2000 - 0.1600)) * ((5.51169 * tas - 173.627) - (6.31891 * tas + 26.2884)) + (6.31891 * tas + 26.2884));
            else
                return (5.51169 * tas - 173.627);
        }
        return NaN;
    }
    getBestMachForRange(altitude, weight) {
        if (this.bestMachForRangeTable && !isNaN(altitude) && !isNaN(weight)) {
            this.bestMachForRangeTable.overrideInputsValue(altitude, weight);
            return this.bestMachForRangeTable.getValue();
        }
        return NaN;
    }
    getBestMachForEndurance(altitude, weight) {
        if (this.bestMachForEnduranceTable && !isNaN(altitude) && !isNaN(weight)) {
            this.bestMachForEnduranceTable.overrideInputsValue(altitude, weight);
            return this.bestMachForEnduranceTable.getValue();
        }
        return NaN;
    }
    computeOptimumRangeAltitude(weight) {
        if (!isNaN(weight)) {
            return 0.00000550716 * Math.pow(weight, 2) - 0.990018 * weight + 72315.1;
        }
        return NaN;
    }
    computeOptimumRangeMach(weight) {
        if (!isNaN(weight)) {
            return 0.845;
        }
        return NaN;
    }
    getMaximumEnduranceAltitude(weight) {
        if (this.maximumEnduranceAltitudeTable && !isNaN(weight)) {
            this.maximumEnduranceAltitudeTable.overrideInputsValue(weight);
            return this.maximumEnduranceAltitudeTable.getValue();
        }
        return NaN;
    }
}
customElements.define("fa18-acc-ddi-fpas", FA18ACC_DDI_FPAS);
//# sourceMappingURL=FA18ACC_DDI_FPAS.js.map