const CargoService = class CargoService {
    constructor() {
        this.MissionCompleteMaximumRadius = 3;

        this.EarthRadius = 6371;
        this.NauticulMileDivisor = 1.852;
        this.StartingLatitude = null;
        this.StartingLongitude = null;
        this.CurrentJob = null;
        this.Airports = [];
        this.MissionPagination = new Pagination();

        this.TotalDistanceTraveledInNm = 0;
        this.TotalMissionsCompleted = 0;

        this.HasTakenOffWithJob = false;
        this.HasLandedWithJob = false;
        this.HasParkedWithJob = false;
        this.HasCompletedJob = false;
    }

    reset() {
        this.StartingLatitude = null;
        this.StartingLongitude = null;
        this.CurrentJob = null;
        this.Airports = [];
        this.MissionPagination = new Pagination();
    }

    getAirportById(airportId) {
        return this.Airports.find(x => x.Id == airportId);
    }

    setupLocation(currentLatitude, currentLongitude) {
        this.StartingLatitude = currentLatitude;
        this.StartingLongitude = currentLongitude;

        return this;
    }

    setupAirports() {
        if (this.StartingLatitude == -1 || this.StartingLongitude == -1) {
            console.error("Failed to access current position - ensure this is set correctly.");
            return;
        }

        this.Airports = [];

        runways.forEach((runway, index) => {
            let airportBuilder = AirportBuilder.create()
                .withId(index)
                .withIcao(runway.icao)
                .withCountry(runway.country)
                .withLatitude(runway.gps[0])
                .withLongitude(runway.gps[1])
                .withRunwayLength(runway.rwy)
                .withDistanceFromStart(this.getDistanceBetweenTwoPointsInNauticalMiles(
                    this.StartingLatitude,
                    this.StartingLongitude,
                    runway.gps[0],
                    runway.gps[1]
                ))
                .withCargoItem(this.getRandomCargoItem());

            this.Airports.push(airportBuilder.build());
        })

        this.Airports.sort((a, b) => a.DistanceFromStart - b.DistanceFromStart);

        return this;
    }

    setupPersonalData(
        smallDistanceTraveledInNm,
        millionDistanceTraveledInNm,
        totalMissionsCompleted
    ) {
        this.TotalMissionsCompleted = +totalMissionsCompleted;
        this.TotalDistanceTraveledInNm = +(
            smallDistanceTraveledInNm +
            (millionDistanceTraveledInNm * 1000000)
        );

        return this;
    }

    getCurrentProgress(currentLatitude, currentLongitude) {
        let currentDistanceFromArrival = this
            .getDistanceBetweenTwoPointsInNauticalMiles(
                currentLatitude,
                currentLongitude,
                this.CurrentJob.ArrivalAirport.Latitude,
                this.CurrentJob.ArrivalAirport.Longitude
            );

        // 3 mile completion radius so 100% there
        if(currentDistanceFromArrival < 3) {
            currentDistanceFromArrival = 0;
        }

        let percentage = 
            currentDistanceFromArrival / this.CurrentJob.ArrivalAirport.DistanceFromStart
            * 100;

        return {
            percentage: Math.ceil(100 - this.clamp(percentage, 0, 100)),
            distance: Math.floor(currentDistanceFromArrival),
        };
    }

    clamp(num, min, max) {
        if (num < min) {
            return min;
        }

        if (num > max) {
            return max;
        }

        return num;
    }

    getDistanceBetweenTwoPointsInNauticalMiles(aLatitude, aLongitude, bLatitude, bLongitude) {
        let [
            aLatitudeRadial,
            aLongitudeRadial,
            bLatitudeRadial,
            bLongitudeRadial,
        ] = [
                this.degreeToRadial(aLatitude),
                this.degreeToRadial(aLongitude),
                this.degreeToRadial(bLatitude),
                this.degreeToRadial(bLongitude),
            ];

        let distance = Math.acos(
            Math.sin(aLatitudeRadial) *
            Math.sin(bLatitudeRadial) +
            Math.cos(aLatitudeRadial) *
            Math.cos(bLatitudeRadial) *
            Math.cos(aLongitudeRadial - bLongitudeRadial)
        ) * this.EarthRadius;

        let distanceInNm = (distance / this.NauticulMileDivisor).toFixed(0);

        return distanceInNm;
    }

    degreeToRadial(degree) {
        return (degree * Math.PI) / 180;
    }

    getAirportsWithinRadiusRange(
        minRadiusInNauticalMiles,
        maxRadiusInNauticalMiles
    ) {
        return this.Airports.filter(x => 
            +x.DistanceFromStart > +minRadiusInNauticalMiles &&
            +x.DistanceFromStart < +maxRadiusInNauticalMiles
        )
    }

    getRandomCargoItem() {
        let randomIndex = Math.floor(Math.random() * cargo_items.length);
        return cargo_items[randomIndex];
    }
}